
/* $Id: postgr.cc,v 1.5 1997/06/27 00:19:32 Jacek Exp $
 *
 * Copyright 1990 Jacek Konieczny
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation, and
 * that the name of Jacek Konieczny not be used in advertising or publicity
 * pertaining to distribution of the software without specific, written prior
 * permission. Jacek Konieczny makes no representations about the suitability
 * of this software for any purpose. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
 * AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Author:	Jacek Konieczny
 *
 *	  email: jajcus@free.polbox.pl
 *                  Currently (June '97) I have very limited Internet
 *                  access so I may sometimes be unavailable this way
 *                  and the address may change in near future.
 *          
 *   snail-mail:
 *
 *		Jacek Konieczny
 *		ul. Kownackiej 2	
 *		44-109 Gliwice
 *		Poland
 */

#include "dbbrowse.h"
#include <iostream.h>
#include <math.h>
#include "postgr.h"

//////////////////////////////////////////////////////////
//        POSTGRES INTERFACE for DBBrowser              //
//////////////////////////////////////////////////////////

DBase::~DBase(){
  if (result) PQclear(result);
  if (catalogue) PQclear(catalogue);
  if (tuple_oids) delete[] tuple_oids;
  if (field_names) delete[] field_names;
  if (types) delete[] types;
  if (field_types) delete[] field_types;
  PQfinish(conn);
}

///////////////////////////////////////////////////

DBase::DBase(String dbName){

  result=NULL;
  catalogue=NULL;
  status=0;
  tuple_oids=NULL;
  field_names=NULL;
  field_types=NULL;
  types=NULL;
  no_rows=no_cols=no_oids=0;

  conn = PQsetdb(NULL,NULL,NULL,NULL,(const char *)dbName);
  if (PQstatus(conn) == CONNECTION_BAD) {
    clog<<"Connection to database "<<dbName<<" failed!\n";
    clog<<PQerrorMessage(conn);
    status=1; 
    return;
  }
  DEBUG_CODE(
    clog<<"Connection to database succsessful!\n"
        <<"Connected to database:\t"<<PQdb(conn)<<'\n'
  	<<"\t\tat host:\t"<<PQhost(conn)<<'\n'
  	<<"\t\tthrow port:\t"<<PQport(conn)<<'\n'
  	<<"\t\toptions:\t"<<PQoptions(conn)<<'\n'
  	<<"\t\tpgtty:\t"<<PQtty(conn)<<"\n\n"
  	    );
  		
  if (get_types()) status=1;
}

///////////////////////////////////////////////////

PGresult * DBase::send_query(const String& query){

  DEBUG_CODE(clog<<"Query:\n"<<query<<" ;\n");
  PQexec(conn,(const char *)query);
}

///////////////////////////////////////////////////

DBase::error_code DBase::begin_transaction(){

  DEBUG_MSG("starting transaction...\n"); 
  PGresult * res=send_query("BEGIN");
  if (!res || PQresultStatus(res) != PGRES_COMMAND_OK){
    clog<<"failed to begin a transaction\n";
    return error;
  }  
  return no_error;
}  

///////////////////////////////////////////////////

DBase::error_code DBase::declare_cursor(const String& name,const String& query){

  DEBUG_MSG("declaring cursor...\n");
  PGresult * res=send_query("DECLARE "+name+" BINARY CURSOR FOR "+query);
  if (!res || PQresultStatus(res) != PGRES_COMMAND_OK){
    clog<<"failed to declare cursor!\n";
    end_transaction();
    return error;
  }  
  return no_error;
}  

///////////////////////////////////////////////////

DBase::error_code DBase::close_cursor(const String& name){

  DEBUG_MSG("closing cursor...\n");
  PGresult * res=send_query("CLOSE "+name);
  if (!res || PQresultStatus(res) != PGRES_COMMAND_OK){
    clog<<"failed to close the cursor\n";
    end_transaction();
    return error;
  }  
  return no_error; 
}

///////////////////////////////////////////////////

DBase::error_code DBase::end_transaction(){

  DEBUG_MSG("ending the transaction...\n");
  PGresult * res=send_query("END");
  if (!res || PQresultStatus(res) != PGRES_COMMAND_OK){
    clog<<"failed to end the transaction\n";
    return error;
  }  
  return no_error;
}

///////////////////////////////////////////////////

DBase::error_code DBase::get_types(){

  DEBUG_MSG("Quering for type list...\n");
  error_code stat;
  stat=begin_transaction();
  if (stat) return stat;
  stat=declare_cursor("cur","SELECT oid,typname,typbyval FROM pg_type");
  if (stat) return stat;
  PGresult * res=send_query("FETCH ALL IN cur");
  if (!res){
    clog<<"Couldn't send query\n";
    clog<<PQstatus(conn)<<'\n';
    close_cursor("cur");
    end_transaction();
    return fatal_error;
  }
  if (PQresultStatus(res)!=PGRES_TUPLES_OK){
    PQclear(res);
    close_cursor("cur");
    end_transaction();
    return error;
  }  
  no_types=PQntuples(res);
  types=new TypeParams[no_types]; 
  Oid char_type=0;
  for(int i=0;i<no_types;i++){
    types[i].oid=*(Oid *)(void *)PQgetvalue(res,i,0);
    types[i].name=PQgetvalue(res,i,1);
  }
  close_cursor("cur");
  end_transaction();
  return no_error;
}


///////////////////////////////////////////////////

String DBase::tuple_oid(int no){
char buf[10];

  sprintf(buf,"%i",tuple_oids[no]);
  return buf;
}  

///////////////////////////////////////////////////

DBase::error_code DBase::init_table(const String& table){

  DEBUG_MSG("Retriving field names...\n");
  String query_cmd="SELECT * FROM "+table+" WHERE oid = 0";
  PGresult *res=send_query(query_cmd);
  if (!res){
    clog<<"Couldn't send query!\n";
    clog<<PQstatus(conn)<<'\n';
    return fatal_error;
  }  
  int stat=PQresultStatus(res);
  if (stat!=PGRES_TUPLES_OK){
    clog<<"Error while querying table!\n";
    clog<<PQcmdStatus(res)<<'\n';
    PQclear(res);
    return error;
  }
  if (field_names) delete[] field_names;
  if (field_types) delete[] field_types;
  no_cols=PQnfields(res);
  field_names=new String [no_cols];
  field_types=new Oid [no_cols];
  for(int i=0;i<no_cols;i++){
    field_names[i]=PQfname(res,i);
    field_types[i]=PQftype(res,i);
  }  
  PQclear(res);
  table_name=table;  
  return no_error;
}

///////////////////////////////////////////////////

DBase::error_code DBase::get_table(const String &query){

  if (table_name=="") return fatal_error;
  DEBUG_MSG("Retrieving oids...\n");

  error_code s=begin_transaction();
  if (s) return s;
  
  s=declare_cursor("cur",query);
  if (s) return s;
 
  PGresult *res=send_query("FETCH ALL IN cur");
  if (!res){
    clog<<"Couldn't send query!\n";
    clog<<PQstatus(conn);
    close_cursor("cur");
    end_transaction();
    return fatal_error;
  }  
  int stat=PQresultStatus(res);
  if (stat!=PGRES_TUPLES_OK){
    clog<<"Error while querying table!\n";
    clog<<PQcmdStatus(res)<<'\n';
    PQclear(res);
    close_cursor("cur");
    end_transaction();
    return error;
  }
  if (tuple_oids) delete[] tuple_oids;
  no_rows=PQntuples(res);
  tuple_oids=new Oid [no_rows+10];
  for(int i=0;i<no_rows;i++)
    tuple_oids[i]=*(Oid *)(void *)PQgetvalue(res,i,0);
  PQclear(res);
  close_cursor("cur");
  end_transaction();
  no_oids=no_rows+10;
  return no_error;
}

///////////////////////////////////////////////////

DBase::error_code DBase::get_table(int no_order_fields,const String *f,
                                                         const String *u){

  String query_cmd="SELECT oid";
  for(int i=0;i<no_order_fields;i++){
    if (f[i]==String("")) continue;
    query_cmd+=", "+f[i];
  }  
    
  query_cmd+="\nFROM "+table_name;
  query_cmd+="\nORDER BY ";
  for(int i=0;i<no_order_fields;i++){
    if (f[i]==String("")) continue;
    query_cmd+=f[i];
    if (u[i]!="") query_cmd+=" USING "+u[i];
    query_cmd+=",";
  }  
  query_cmd+="oid";

  return get_table(query_cmd);
}

///////////////////////////////////////////////////

DBase::error_code DBase::search(const String * field_checks,
                          int no_order_fields,const String *f,const String *u){

  String query_cmd="SELECT oid";
  for(int i=0;i<no_order_fields;i++){
    if (f[i]=="") continue;
    query_cmd+=" , "+f[i];
  }  
  query_cmd+="\nFROM "+table_name;
  query_cmd+="\nWHERE";
  for(int i=0;i<no_cols;i++)
    if (field_checks[i]!="")
          query_cmd+="   "+field_names[i]+' '+field_checks[i]+'\n';
  query_cmd+="\nORDER BY ";
  for(int i=0;i<no_order_fields;i++){
    if (f[i]==String("")) continue;
    query_cmd+=f[i];
    if (u[i]!=String("")) query_cmd+=" USING "+u[i];
    query_cmd+=",";
  }  
  query_cmd+="oid\n";

  return get_table(query_cmd);
}

///////////////////////////////////////////////////

DBase::error_code DBase::query_tuple(int no){

  if (no>=no_tuples()) return error;
  DEBUG_MSG("Querying tuple...\n");
  String query_cmd="SELECT * FROM "+table_name;
  query_cmd+=" WHERE oid ="+tuple_oid(no);
  PGresult *res=send_query(query_cmd);
  if (!res){
    clog<<"Couldn't send query!\n";
    clog<<PQstatus(conn)<<'\n';
    return fatal_error;
  }  
  int stat=PQresultStatus(res);
  if (stat!=PGRES_TUPLES_OK || PQntuples(res)<1 ){
    clog<<"Error while querying table!\n";
    clog<<PQcmdStatus(res)<<'\n';
    PQclear(res);
    return error;
  }
  if (result) PQclear(result);
  result=res;
  return no_error;
}

///////////////////////////////////////////////////

DBase::error_code DBase::change(int tuple,int field,const String& new_value){

  DEBUG_MSG("Updating table...\n");
  String query_cmd="UPDATE "+table_name+'\n';
  query_cmd+="SET ";
  query_cmd+=field_names[field];
  query_cmd+=" = ";
  if (new_value!="")
    query_cmd+=str2sql(new_value,field_types[field]);
  else query_cmd+="NULL\n";
  query_cmd+=" WHERE oid = "+tuple_oid(tuple);
  PGresult *res=send_query(query_cmd);
  if (!res){
    clog<<"Couldn't send query!\n";
    clog<<PQstatus(conn)<<'\n';
    return fatal_error;
  }  
  int stat=PQresultStatus(res);
  if (stat!=PGRES_COMMAND_OK){
    clog<<"Error while querying table!\n";
    clog<<PQcmdStatus(res)<<'\n';
    PQclear(res);
    return error;
  }
  PQclear(res);
  return no_error;
}

///////////////////////////////////////////////////

DBase::error_code DBase::add(String * values){

  DEBUG_MSG("Adding to table...\n");
  String query_cmd="INSERT INTO "+table_name;
  query_cmd+='\n';
  query_cmd+="VALUES (\n";
  int no=no_fields();
  for(int f=0;f<no;f++){
    if (values[f]!="")
      query_cmd+="   "+str2sql(values[f],field_types[f]);
    else query_cmd+="   NULL"; 
    if (f<no-1) query_cmd+=",\n";
    else query_cmd+=")\n";
  }  
  PGresult *res=send_query(query_cmd);
  if (!res){
    clog<<"Couldn't send query!\n";
    clog<<PQstatus(conn);
    return fatal_error;
  }  
  int stat=PQresultStatus(res);
  if (stat!=PGRES_COMMAND_OK){
    clog<<"Error while querying table!\n";
    clog<<PQcmdStatus(res)<<'\n';
    PQclear(res);
    return error;
  }
  int oid=atoi(PQoidStatus(res));
  PQclear(res);
  if (!tuple_oids){
    tuple_oids=new Oid [10];
    no_oids=10;
  }
  else if (no_oids>=no_rows){
    Oid * new_oids=new Oid [no_rows+10];
    memcpy(new_oids,tuple_oids,no_oids*sizeof(Oid));
    delete tuple_oids;
    tuple_oids=new_oids;
    no_oids=no_rows+10;
  }
  tuple_oids[no_rows++]=oid;
  return no_error;
}

///////////////////////////////////////////////////

DBase::error_code DBase::remove(int no){

  DEBUG_MSG("Removing from table...\n");
  String query_cmd="DELETE FROM "+table_name;
  query_cmd+="\n   WHERE oid = "+tuple_oid(no);
  PGresult *res=send_query(query_cmd);
  if (!res){
    clog<<"Couldn't send query!\n";
    return fatal_error;
  }  
  int stat=PQresultStatus(res);
  if (stat!=PGRES_COMMAND_OK){
    clog<<"Error while querying!\n";
    clog<<PQcmdStatus(res)<<'\n';
    PQclear(res);
    return error;
  }
  if (no!=no_rows-1)
    memmove(tuple_oids+no,tuple_oids+no+1,(no_rows-no-1)*sizeof(Oid));
  no_rows--;
  return no_error;
}

///////////////////////////////////////////////////

String DBase::get_value(int col){
  if (!result) return ""; 
  return PQgetvalue(result,0,col);
}


///////////////////////////////////////////////////

int DBase::get_table_catalogue(){

  DEBUG_MSG("Quering for class catalogue...\n");
  PGresult * res=send_query("SELECT relname FROM pg_class WHERE relkind = 'r' and relname !~ '^pg_' and relname !~ '^Inv' ORDER BY relname");
  if (!res){
    clog<<"Couldn't send query!\n";
    return 0;
  }  
  if (PQresultStatus(res)!=PGRES_TUPLES_OK) return 0;
  int no=PQntuples(res);
  if (catalogue) PQclear(catalogue);
  catalogue=res;
  DEBUG_CODE(clog<<"tuples got: "<<no<<'\n');
  return no;
}

int DBase::get_operator_catalogue(Oid result,Oid left,Oid right){
char buf[16];

  DEBUG_MSG("Quering for operator catalogue...\n");
  String query_cmd="SELECT oprname FROM pg_operator\nWHERE ";
  sprintf(buf,"%d",result);
  query_cmd+="oprresult = ";
  query_cmd+=buf;
  sprintf(buf,"%d",left);
  query_cmd+="\n and oprleft = ";
  query_cmd+=buf;
  sprintf(buf,"%d",right);
  query_cmd+="\n and oprright = ";
  query_cmd+=buf;
  query_cmd+="\nGROUP BY oprname";

  PGresult * res=send_query(query_cmd);
  if (!res){
    clog<<"Couldn't send query!\n";
    clog<<PQstatus(conn)<<'\n';
    return 0;
  }  
  if (PQresultStatus(res)!=PGRES_TUPLES_OK) return 0;
  int no=PQntuples(res);
  if (catalogue) PQclear(catalogue);
  catalogue=res;
  DEBUG_CODE(clog<<"tuples got: "<<no<<'\n');
  return no;
}

///////////////////////////////////////////////////

void DBase::close_catalogue(){
  if (catalogue) PQclear(catalogue);
  catalogue=NULL;
}


///////////////////////////////////////////////////

String DBase::read_catalogue(int ind){
  if (!catalogue) return "";
  return PQgetvalue(catalogue,ind,0);
}
  
