#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <linux/major.h>

#define SCSI_IOCTL_START_STOP 8			/* Start or stop unit */

#define SD_IOCTL_IDLE 4746	/* get idle time */

#define DEBUG 0

char lockfn[64];

void handler()
{
  unlink(lockfn);
  exit(0);
}

int main(int argc, char *argv[])
{
  int fd;
  struct stat statbuf;

  if (argc < 2 || argc > 3)
  {
    fprintf(stderr, 
	    "usage: %s device [timeout]\n"
	    "  where timeout is the time until motor off in seconds\n"
	    "  to idle the disk immediately, use a timeout of zero\n",
	    argv[0]);
    exit(1);
  }

  if ((fd = open(argv[1], O_RDWR)) < 0)
  {
    perror(argv[1]);
    exit(1);
  }

  if ((fstat(fd, &statbuf)) < 0)
  {
    perror(argv[1]);
    close(fd);
    exit(1);
  }

  if (!S_ISBLK(statbuf.st_mode)
      || ( (statbuf.st_rdev >> 8) & (255 != SCSI_DISK_MAJOR) ) )
  {
    fprintf(stderr, "%s is not a SCSI block device\n", argv[1]);
    close(fd);
    exit(1);
  }

  if (argc == 2)
  {
    long last;

    /* Report idle time */

    if ((last = ioctl(fd, SD_IOCTL_IDLE, &last)) < 0)
    {
      perror(argv[1]);
      close(fd);
      exit(1);
    }
    if (last == 0)
      printf("%s has not been accessed\n", argv[1]);
    else
      printf("%s has been idle for %ld second%s\n", argv[1],
	     last, (last==1 ? "":"s"));
  }
  else
  {
    long timeout;

    if ((timeout = atol(argv[2])) < 0)
    {
      fprintf(stderr, "timeout may not be negative\n");
      close(fd);
      exit(1);
    }

    if (timeout == 0)
    {
      long off = 0;

      /* Spin down disk immediately */

      if (ioctl(fd, SCSI_IOCTL_START_STOP, &off) < 0)
      {
	perror(argv[1]);
	close(fd);
	exit(1);
      }
    }
    else
    {
      FILE *fp;
      int dev;
      long last;
      pid_t pid;

      /* Spin down disk after an idle period */
      if (timeout < 30)
	fprintf(stderr, "Warning: timeouts less than 30 seconds are really not such a good idea\n");

      /* This is ugly, but I guess there is no portable way to do this */
      dev = ((statbuf.st_rdev >> 4) & 15) + 'a';

      sprintf(lockfn, "/var/run/scsi-idle.sd%c.pid", dev);

      if ((fp = fopen(lockfn, "r")) != NULL)
      {
	fscanf(fp, "%d", &pid);
	fclose(fp);

	kill(pid, SIGTERM);

	unlink(lockfn);
      }

      switch (fork())
      {
      case -1:
	perror("fork failed");
	close(fd);
	exit(1);

      case 0:
	signal(SIGINT, handler);
	signal(SIGTERM, handler);

	if ((fp = fopen(lockfn, "w")) == NULL)
	{
	  perror(lockfn);
	  close(fd);
	  exit(1);
	}
	fprintf(fp, "%d\n", getpid());
	fclose(fp);

#if 1
	nice(10);
#endif

	for (;;)
	{
	  if ((last = ioctl(fd, SD_IOCTL_IDLE, &last)) < 0)
	  {
	    perror(argv[1]);
	    close(fd);
	    exit(1);
	  }
#if DEBUG
	  if (last == 0)
	    printf("%s has not been accessed\n", argv[1]);
	  else
	    printf("%s has been idle for %d second%s\n", argv[1],
		   last, (last==1 ? "":"s"));
#endif
	  
	  if (last > timeout)
	  {
	    long off = 0;
	    
	    if (ioctl(fd, SCSI_IOCTL_START_STOP, &off) != 0)
	    {
	      perror(argv[1]);
	      close(fd);
	      exit(1);
	    }
	    last = 0;
	  }
	  
#if DEBUG
	  printf("sleeping for %d seconds\n", timeout - last + 1);
#endif
	  sleep(timeout - last + 1);
	}
	/* notreached */
       
      default:
	printf("%s: idle daemon started, timeout is %ld seconds\n", 
	       argv[1], timeout);
	/* Ok, done */
	break;
      }
    }
  }

  close(fd);
  exit(0);
}
